/* betaFunctions.cpp */

#include "fp.h"


bool beta(fp& z, fp& a, fp& b)
{
	fp  Ga, Gb, Gab;
	
	Gamma(Ga, a);
	Gamma(Gb, b);
	Gamma(Gab, a+b);
	
	z = Ga*Gb/Gab;
	
	return true;
	
}/* beta */


/*
From "Numerical Recipes", p. 136:

Let the continued fraction be:

f = b[0] + a[1]/b[1]+ a[2]/b[2]+ a[3]/b[3]+ ...

Then the two term recursion relation is:

p[n] = b[n]*p[n-1] + a[n]*p[n-2]

q[n] = b[n]*q[n-1] + a[n]*q[n-2]    n = 1, 2, ....

where

p[-1] = 1
p[0] = b[0]
q[-1] = 0
q[0] = 1

and

f = lim(n->inf) p[n]/q[n]

*/


bool iBeta(fp& z, fp& a, fp& b, fp& x)
{
	fp			pA, pB, pC, qA, qB, qC;
	fp			A, B;
	INT32		n, m;
	fp			f, fLast;
	fp			temp;
	bool		xIsLow;
	fp			Beta, theNum;
	fp			one=1.;
	
	z = 0;
	
	if(!x.i.n)
		return true;
	
	if(x.i.n<0 || x>one)
		return false;
	
	beta(Beta, a, b);
	if(x==one)
	{
		z = Beta;
		return true;
	}
	
	theNum = pow(x, a)*pow(1.-x, b);
	
	if(x<=(a+1)/(a+b+1))
		xIsLow = true;
	else
	{
		xIsLow = false;
		temp = a;
		a = b;
		b = temp;
		x = 1 - x;
	}
	
	// initializations
	pA = 1.; // p[n-2] for n=1
	pB = 0; // p[n-1] for n=1
	qA = 0; // q[n-2] for n=1
	qB = 1.; // q[n-1] for n=1
	
	A = 1.; // a[n] for n=1
	B = 1.; // b[n] for n=1
	
	// do n=1 recursion
	pC = B*pB + A*pA; // pC = p[n+2] for n=0
	qC = B*qB + A*qA; // qC = q[n+2] for n=0
	
	fLast = 0; // to be used for end game
	
	// update for next iteration
	pA = pB;
	pB = pC;
	qA = qB;
	qB = qC;
	
	// then the remaining until practical convergence
	for (n=2; ;n++)
	{
		if(2*(n/2)==n)
		{
			m = (n-2)/2;
			A = -(a+m)*(a+b+m)*x/((a+2*m)*(a+2*m+1));
		}
		else
		{
			m = (n-1)/2;
			A = m*(b-m)*x/((a+2*m-1)*(a+2*m));
		}
		
		// do nth recursion
		pC = B*pB + A*pA;
		qC = B*qB + A*qA;
		f = pC/qC;
		if(f==fLast)
			break; // exit the for loop
		fLast = f;
		pA = pB;
		pB = pC;
		qA = qB;
		qB = qC;
	}
	
	z = f*theNum/(a*Beta);
	
	if(!xIsLow)
		z = 1 - z;
	
	// now z is I(a, b, x), perhaps we should output that also
		
	z = Beta*z;
	
	return true;
	
}/* iBeta */

